import { AttachmentList } from "@mail/core/common/attachment_list";
import { Composer } from "@mail/core/common/composer";
import { ImStatus } from "@mail/core/common/im_status";
import { MessageInReply } from "@mail/core/common/message_in_reply";
import { MessageLinkPreviewList } from "@mail/core/common/message_link_preview_list";
import { MessageNotificationPopover } from "@mail/core/common/message_notification_popover";
import { MessageReactionMenu } from "@mail/core/common/message_reaction_menu";
import { MessageReactions } from "@mail/core/common/message_reactions";
import { Poll } from "@mail/core/common/poll";
import { PollResult } from "@mail/core/common/poll_result";
import { RelativeTime } from "@mail/core/common/relative_time";
import { htmlToTextContentInline } from "@mail/utils/common/format";
import { isEventHandled, markEventHandled } from "@web/core/utils/misc";
import { renderToElement } from "@web/core/utils/render";

import {
    Component,
    onMounted,
    toRaw,
    useChildSubEnv,
    useEffect,
    useRef,
    useState,
    useSubEnv,
} from "@odoo/owl";

import { ActionSwiper } from "@web/core/action_swiper/action_swiper";
import { isMobileOS } from "@web/core/browser/feature_detection";
import { Dropdown } from "@web/core/dropdown/dropdown";
import { useDropdownState } from "@web/core/dropdown/dropdown_hooks";
import { _t } from "@web/core/l10n/translation";
import { usePopover } from "@web/core/popover/popover_hook";
import { useChildRef, useService } from "@web/core/utils/hooks";
import { createElementWithContent } from "@web/core/utils/html";
import { getOrigin, url } from "@web/core/utils/urls";
import { useMessageActions } from "./message_actions";
import { discussComponentRegistry } from "./discuss_component_registry";
import { NotificationMessage } from "./notification_message";
import { useForwardRefsToParent, useLongPress } from "@mail/utils/common/hooks";
import { ActionList } from "@mail/core/common/action_list";
import { loadCssFromBundle } from "@mail/utils/common/misc";
import { MessageContextMenu } from "./message_context_menu";

/**
 * @typedef {Object} Props
 * @property {boolean} [hasActions=true]
 * @property {boolean} [highlighted]
 * @property {function} [onParentMessageClick]
 * @property {import("models").Message} message
 * @property {boolean} [squashed]
 * @property {import("models").Thread} [thread]
 * @property {ReturnType<import('@mail/utils/common/hooks').useMessageSelection>} [messageSelection]
 * @property {ReturnType<import('@mail/core/common/message_search_hook').useMessageSearch>} [messageSearch]
 * @property {String} [className]
 * @extends {Component<Props, Env>}
 */
export class Message extends Component {
    // This is the darken version of #71639e
    static SHADOW_LINK_COLOR = "#66598f";
    static SHADOW_HIGHLIGHT_COLOR = "#e99d00bf";
    static SHADOW_LINK_HOVER_COLOR = "#564b79";
    static components = {
        ActionList,
        ActionSwiper,
        AttachmentList,
        Composer,
        Dropdown,
        ImStatus,
        MessageContextMenu,
        MessageInReply,
        MessageLinkPreviewList,
        MessageReactions,
        Popover: MessageNotificationPopover,
        Poll,
        PollResult,
        RelativeTime,
        NotificationMessage,
    };
    static defaultProps = {
        hasActions: true,
        showDates: true,
    };
    static props = [
        "asCard?",
        "hasActions?",
        "onParentMessageClick?",
        "message",
        "messageSelection?",
        "messageRefs?",
        "previousMessage?",
        "squashed?",
        "thread?",
        "messageSearch?",
        "className?",
        "showDates?",
        "isFirstMessage?",
        "isReadOnly?",
    ];
    static template = "mail.Message";

    setup() {
        super.setup();
        this.store = useService("mail.store");
        this.popover = usePopover(this.constructor.components.Popover, { position: "top" });
        this.state = useState({
            isHovered: false,
            isClicked: false,
            expandOptions: false,
            emailHeaderOpen: false,
        });
        this.rightClickDropdownState = useDropdownState({
            onClose: () => this.props.messageSelection.clearSelected(),
        });
        this.rightClickAnchor = useChildRef("rightClickAnchor");
        /** @type {ShadowRoot} */
        this.shadowRoot;
        this.root = useRef("root");
        if (isMobileOS()) {
            useLongPress("root", {
                action: () => this.openMobileActions(),
                predicate: () => !this.isEditing,
            });
        }
        useForwardRefsToParent("messageRefs", (props) => props.message.id, this.root);
        this.messageBody = useRef("body");
        this.messageActions = useMessageActions({
            message: () => this.message,
            thread: () => this.props.thread,
        });
        this.shadowBody = useRef("shadowBody");
        this.dialog = useService("dialog");
        this.ui = useService("ui");
        this.openReactionMenu = this.openReactionMenu.bind(this);
        this.optionsDropdown = useDropdownState();
        useSubEnv({ inMessage: true });
        useChildSubEnv({
            message: this.props.message,
            alignedRight: this.isAlignedRight,
        });
        onMounted(() => {
            if (this.shadowBody.el) {
                this.shadowRoot = this.shadowBody.el.attachShadow({ mode: "open" });
                const color = this.store.isOdooWhiteTheme ? "dark" : "white";
                loadCssFromBundle(this.shadowRoot, "mail.assets_message_email");
                const shadowStyle = document.createElement("style");
                shadowStyle.textContent = `
                    * {
                        background-color: transparent !important;
                        color: ${color} !important;
                    }
                    a, a * {
                        color: ${this.constructor.SHADOW_LINK_COLOR} !important;
                    }
                    a:hover, a *:hover {
                        color: ${this.constructor.SHADOW_LINK_HOVER_COLOR} !important;
                    }
                    .o-mail-Message-searchHighlight {
                        background: ${this.constructor.SHADOW_HIGHLIGHT_COLOR} !important;
                    }
                `;
                if (!this.store.isOdooWhiteTheme) {
                    this.shadowRoot.appendChild(shadowStyle);
                }
                const ellipsisStyle = document.createElement("style");
                ellipsisStyle.textContent = `
                    .o-mail-ellipsis {
                        min-width: 2.7ch;
                        background-color: ButtonFace;
                        border-radius: 50rem;
                        border: 0;
                        display: block
                        font: -moz-button;
                        font-size: .75rem;
                        font-weight: 500;
                        line-height: 1.1;
                        cursor: pointer;
                        padding: 0 4px;
                        vertical-align: top;
                        color #ffffff;
                        text-decoration: none;
                        text-align: center;
                        &:hover {
                            background-color: -moz-buttonhoverface;
                        }
                    }
                `;
                this.shadowRoot.appendChild(ellipsisStyle);
            }
        });
        useEffect(
            () => {
                if (this.shadowBody.el) {
                    const bodyEl = createElementWithContent(
                        "span",
                        this.message.showTranslation
                            ? this.message.richTranslationValue
                            : this.props.messageSearch?.highlight(this.message.richBody) ??
                                  this.message.richBody
                    );
                    this.prepareMessageBody(bodyEl);
                    this.shadowRoot.appendChild(bodyEl);
                    return () => {
                        this.shadowRoot.removeChild(bodyEl);
                    };
                }
            },
            () => [
                this.message.showTranslation,
                this.message.richTranslationValue,
                this.props.messageSearch?.searchTerm,
                this.message.richBody,
                this.isEditing,
            ]
        );
        useEffect(
            () => {
                if (!this.isEditing) {
                    this.prepareMessageBody(this.messageBody.el);
                }
            },
            () => [this.isEditing, this.message.richBody]
        );
    }

    computeActions() {
        const allActions = this.messageActions.actions;
        const quickActions = allActions.slice(
            0,
            allActions.length > this.quickActionCount
                ? this.quickActionCount - 1
                : this.quickActionCount
        );
        const moreActions =
            allActions.length > this.quickActionCount
                ? allActions.slice(this.quickActionCount - 1)
                : false;
        const moreAction = moreActions?.length
            ? this.messageActions.more({
                  actions: moreActions,
                  dropdownMenuClass: "o-mail-Message-moreMenu",
                  dropdownPosition: this.isAlignedRight
                      ? this.message.threadAsNewest
                          ? "left-end"
                          : "left-start"
                      : this.message.threadAsNewest
                      ? "right-end"
                      : "right-start",
                  name: this.expandText,
              })
            : undefined;
        const actions = moreAction ? [...quickActions, moreAction] : quickActions;
        if (this.isAlignedRight) {
            actions.reverse();
        }
        this.state.moreAction = moreAction;
        this.quickActions = quickActions;
        this.actions = actions;
    }

    get attClass() {
        return {
            "user-select-none o-isMobileOS": isMobileOS(),
            [this.props.className]: true,
            "o-card p-2 ps-1 mx-1 mt-1 mb-1 border border-dark rounded-2": this.props.asCard,
            "pt-1": !this.props.asCard && !this.props.squashed,
            "o-pt-0_5": !this.props.asCard && this.props.squashed,
            "o-selfAuthored": this.message.isSelfAuthored && !this.env.messageCard,
            "o-selected":
                this.props.message.composerAsReplyToMessage?.thread.eq(this.props.thread) ||
                this.props.messageSelection?.isSelected(this.props.message),
            "o-squashed": this.props.squashed,
            "mt-1":
                !this.props.squashed &&
                this.props.thread &&
                !this.env.messageCard &&
                !this.props.asCard,
            "px-1": this.env.inChatWindow,
            "o-actionMenuMobileOpen": this.ui.isSmall && this.optionsDropdown.isOpen,
            "o-editing": this.isEditing,
        };
    }

    get authorAvatarAttClass() {
        return {
            "object-fit-contain": this.props.message.author_id?.is_company,
            "object-fit-cover": !this.props.message.author_id?.is_company,
        };
    }

    get authorAvatarUrl() {
        if (
            this.message.message_type &&
            this.message.message_type.includes("email") &&
            !this.message.author_id &&
            !this.message.author_guest_id
        ) {
            return url("/mail/static/src/img/email_icon.png");
        }
        if (this.message.author) {
            return this.message.author.avatarUrl;
        }
        return this.store.DEFAULT_AVATAR;
    }

    get expandText() {
        return _t("Expand");
    }

    get isEditing() {
        return !this.props.isReadOnly && this.props.message.composer;
    }

    get message() {
        return this.props.message;
    }

    /** Max amount of quick actions, including "..." */
    get quickActionCount() {
        if (isMobileOS()) {
            return 1;
        }
        return this.env.inChatWindow || this.env.inMeetingChat ? 2 : 4;
    }

    get showSubtypeDescription() {
        return (
            this.message.subtype_id?.description &&
            this.message.subtype_id.description.toLowerCase() !==
                htmlToTextContentInline(this.message.body || "").toLowerCase()
        );
    }

    get messageTypeText() {
        if (this.props.message.message_type === "notification") {
            return _t("System notification");
        }
        if (this.props.message.message_type === "auto_comment") {
            return _t("Automated message");
        }
        if (this.props.message.message_type === "out_of_office") {
            return _t("Out-of-office message");
        }
        if (
            !this.props.message.isDiscussion &&
            this.props.message.message_type !== "user_notification"
        ) {
            return _t("Note");
        }
        return _t("Message");
    }

    get isActive() {
        return (
            this.state.isHovered ||
            this.state.isClicked ||
            this.emojiPicker?.isOpen ||
            Boolean(this.state.moreAction?.isActive)
        );
    }

    get isAlignedRight() {
        return Boolean(
            this.env.inChatWindow && this.props.message.isSelfAuthored && !this.env.messageCard
        );
    }

    get isMobileOS() {
        return isMobileOS();
    }

    get isPersistentMessageFromAnotherThread() {
        return (
            !this.message.is_transient &&
            !this.message.isPending &&
            this.message.thread &&
            this.message.thread.notEq(this.props.thread)
        );
    }

    get onRightSwipe() {
        if (this.props.thread?.eq(this.store.inbox)) {
            return {
                action: () => this.message.setDone(),
                bgColor: "bg-success",
                icon: "fa-check-circle",
            };
        }
        if (
            this.props.thread?.eq(this.store.history) &&
            this.message.canMarkAsUnread(this.props.thread)
        ) {
            return {
                action: () => this.message.markAsUnread(this.props.thread),
                bgColor: "bg-secondary",
                icon: "fa-eye-slash",
            };
        }
        return undefined;
    }

    get translatedFromText() {
        return _t("(Translated from: %(language)s)", { language: this.message.translationSource });
    }

    get translationFailureText() {
        return _t("(Translation Failure: %(error)s)", { error: this.message.translationErrors });
    }

    onMouseenter() {
        this.state.isHovered = true;
    }

    onMouseleave() {
        this.state.isHovered = false;
        this.state.isClicked = null;
    }

    /**
     * @returns {boolean}
     */
    get shouldDisplayAuthorName() {
        if (!this.env.inChatWindow || this.env.messageCard) {
            return true;
        }
        if (this.message.isSelfAuthored) {
            return false;
        }
        if (this.props.thread.channel?.channel_type === "chat") {
            return false;
        }
        return true;
    }

    async onClickAttachmentUnlink(attachment) {
        await toRaw(attachment).remove();
    }

    /**
     * @param {MouseEvent} ev
     */
    async onClick(ev) {
        if (this.store.handleClickOnLink(ev, this.props.thread)) {
            return;
        }
        if (
            !isEventHandled(ev, "Message.ClickAuthor") &&
            !isEventHandled(ev, "Message.ClickFailure")
        ) {
            if (this.state.isClicked) {
                this.state.isClicked = false;
            } else {
                this.state.isClicked = true;
                document.body.addEventListener(
                    "click",
                    () => {
                        this.state.isClicked = false;
                    },
                    { capture: true, once: true }
                );
            }
        }
    }

    onContextMenu(ev) {
        if (!document.getSelection()?.isCollapsed || isMobileOS()) {
            // text selection by-passes message actions on right-click.
            // Mobile OS long press is handled with useLongPress()
            return;
        }
        const el = this.rightClickAnchor.el;
        el.style.left = ev.clientX + "px";
        el.style.top = ev.clientY + "px";
        this.rightClickDropdownState.open();
        this.props.messageSelection?.setSelected(this.props.message);
        ev.preventDefault();
    }

    /** @param {HTMLElement} bodyEl */
    prepareMessageBody(bodyEl) {
        if (!bodyEl) {
            return;
        }
        const editedEl = bodyEl.querySelector(".o-mail-Message-edited");
        editedEl?.replaceChildren(renderToElement("mail.Message.edited"));
        const channelLinks = bodyEl.querySelectorAll("a.o_channel_redirect");
        this.store.handleValidChannelMention(Array.from(channelLinks));
        for (const el of bodyEl.querySelectorAll(".o_message_redirect")) {
            // only transform links targetting the same database
            if (el.getAttribute("href")?.startsWith(getOrigin())) {
                const message = this.store["mail.message"].get(el.dataset.oeId);
                if (message?.thread?.displayName) {
                    el.classList.add("o_message_redirect_transformed");
                    el.replaceChildren(renderToElement("mail.Message.messageLink", { message }));
                }
            }
        }
    }

    getAuthorAttClass() {
        return { "opacity-50": this.message.isPending };
    }

    getAvatarContainerAttClass() {
        return {
            "opacity-50": this.message.isPending,
            "o-inChatWindow": this.env.inChatWindow,
        };
    }

    exitEditMode() {
        this.message.exitEditMode(this.props.thread);
    }

    onClickNotification(ev) {
        const message = toRaw(this.message);
        if (message.failureNotifications.length > 0) {
            markEventHandled(ev, "Message.ClickFailure");
        }
        this.popover.open(ev.target, { message });
    }

    /** @param {MouseEvent} [ev] */
    openMobileActions(ev) {
        if (!isMobileOS()) {
            return;
        }
        ev?.stopPropagation();
        this.optionsDropdown.open();
    }

    openReactionMenu(reaction) {
        const message = toRaw(this.props.message);
        this.dialog.add(
            MessageReactionMenu,
            { message, initialReaction: reaction },
            { context: this }
        );
    }
}

discussComponentRegistry.add("Message", Message);
