# -*- coding: utf-8 -*-
# Part of Odoo. See LICENSE file for full copyright and licensing details.
# Copyright (C) 2004-2008 PC Solutions (<http://pcsol.be>). All Rights Reserved
from odoo import fields, models, api, _
from odoo.exceptions import ValidationError
from odoo.addons.base.models.ir_model import MODULE_UNINSTALL_FLAG


class AccountJournal(models.Model):
    _inherit = 'account.journal'

    pos_payment_method_ids = fields.One2many('pos.payment.method', 'journal_id', string='Point of Sale Payment Methods')

    @api.constrains('type')
    def _check_type(self):
        methods = self.env['pos.payment.method'].sudo().search([("journal_id", "in", self.ids)])
        if methods:
            raise ValidationError(_("This journal is associated with a payment method. You cannot modify its type"))

    def _check_no_active_payments(self):
        linked_payment_methods = self.env['pos.payment.method'].sudo().search([('journal_id', 'in', self.ids)], limit=1)
        if linked_payment_methods:
            raise ValidationError(_("You can not archive this journal because it is set on the following payment method : %s.", linked_payment_methods.name))

    @api.ondelete(at_uninstall=False)
    def _unlink_journal_except_with_active_payments(self):
        for journal in self:
            journal._check_no_active_payments()

    @api.ondelete(at_uninstall=True)
    def _unlink_journal_cascade_pos_payment_methods(self):
        if self.env.context.get(MODULE_UNINSTALL_FLAG):  # only cascade when switching CoA
            self.pos_payment_method_ids.unlink()
            self.env['pos.config'].search([('journal_id', 'in', self.ids)]).unlink()

    def action_archive(self):
        self._check_no_active_payments()
        return super().action_archive()

    def _get_journal_inbound_outstanding_payment_accounts(self):
        return (
            super()._get_journal_inbound_outstanding_payment_accounts()
            | self.sudo().pos_payment_method_ids.outstanding_account_id.with_env(self.env)
        )

    @api.model
    def _ensure_company_account_journal(self):
        journal = self.search([
            ('code', '=', 'POSS'),
            ('company_id', '=', self.env.company.id),
        ], limit=1)
        if not journal:
            journal = self.create({
                'name': _('Point of Sale'),
                'code': 'POSS',
                'type': 'general',
                'company_id': self.env.company.id,
            })
        return journal
